function rxnList = mapGeneCalls(geneNames,highOrLow,model)
% rxnList = mapGeneCalls(geneNames,highOrLow,model)
% Chris Gowen, April 2012
% 
% Use the rules field of a model and a list of reactions that are either
% 'high' or 'low' to determine which reactions are affected.
%
% Inputs (required):
%  - geneNames      : Cell array of genes names.
%  - highOrLow      : 'high' or 'low', depending on the direction of the
%                       call
%  - model          : Model structure in typical COBRA format. Must have
%                       field 'rules' which has logical statement version
%                       of gene-protein-reaction rules referencing a
%                       boolean matrix x corresponding to gene index. e.g.,
%                       >>model.rules{strmatch('PYK',model.rxns)) 
%                            ans = 
%                            '( x(12)  |  x(828) )'
%
% Output
%  - rxnList        : List of reaction indices that are 'high' or 'low'
%
% This function is provided to accompany the chapter "Linking RNA 
% measurements and proteomics with genome-scale models" in Methods in
% Molecular Biology: Systems Metabolic Engineering, ed. Hal Alper
%
% It is provided for educational purposes only, without any warranty.

    
    % Find the genes that are in the model:
    [~, genes] = ismember(geneNames,model.genes);
    genes = genes(find(genes));

    rxnInd = find(any(model.rxnGeneMat(:,genes),2));
    
    if (strcmp(highOrLow,'high'))
        x = false(size(model.genes));
        x(genes) = true;
        
        rxnsAffected = false(size(model.rxns));
        for i=1:length(rxnInd)
            r=rxnInd(i);
            rxnsAffected(r) = eval(model.rules{r});
        end
        rxnList = find(rxnsAffected);
    
    elseif (strcmp(highOrLow,'low'))
        x = true(size(model.genes));
        x(genes) = false;
        
        rxnsAffected = true(size(model.rxns));
        for i=1:length(rxnInd)
            r=rxnInd(i);
            rxnsAffected(r) = eval(model.rules{r});
        end
        rxnList = find(rxnsAffected==false);
    end


end